//==============================================================================
// Project:		Wealth transfers and their economic effects
// File name:	Table B.5 part a
// Objective: 	Generate the data underpinning the non-pooled component of table B.5
//
//==============================================================================

clear all

************************
*** Data import ***
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

merge m:1 wave using CPI.dta //CPI data
drop if _merge == 2
drop _merge

merge m:1 xwaveid using replicate_weights.dta //replicate weights
drop if _merge == 2
drop _merge 

xtset xwaveid wave

************************
*** Create variables ***
************************

replace oifinha = 0 if oifinha == -10 | oifinha == -1 //inheritances
replace oifinha = . if oifinha < 0
gen bequest = oifinha
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due HH deaths
 
gen net_worth = hwnwip - hwnwin //household net worth

rename wlebf weight_2 //sampling weights
rename wlefj weight_6
rename wlejn weight_10
rename wlenr weight_14

forvalues x = 1/45{ //replicate weights
	rename wlebf`x' r_weight_2_`x'
	rename wlefj`x' r_weight_6_`x'
	rename wlejn`x' r_weight_10_`x'
	rename wlenr`x' r_weight_14_`x'
}

gen child = 1 if hgage < 15 //generate equivalisation scale (OECD modified)
gen adult = 1 if hgage >= 15
egen total_child = total(child), by(hhrhid wave)
egen total_adult = total(adult), by(hhrhid wave)
gen equiv_scale = total_child*0.3 + total_adult*0.5 + 0.5 //0.3 for children, 1 for first adult and 0.5 for all subsequent adults

egen total_bequest = total(bequest), by(hhrhid wave)
gen bequest_e = total_bequest * 100 / (equiv_scale*CPI) //real equivalised inheritances

gen net_worth_e = net_worth * 100 / (equiv_scale*CPI) //real equivalised net worth

gen net_worth_bequest_e = net_worth_e + F2.bequest_e + F3.bequest_e + F4.bequest_e

gen F2_0_bequest_e = F2.bequest_e + F3.bequest_e + F4.bequest_e

gen F2_0_bequest_e_positive = 0
replace F2_0_bequest_e_positive = 1 if F2_0_bequest_e > 0 //flag for an inheritance

gen flag = 1

drop if net_worth_bequest_e == . //drop cases where the person not in both waves, due to death or moving abroad (the weights don't adjust for this)

*****************************************************
*** Create program for jackknifing wealth measures **
*****************************************************

program jackknife_diff, eclass properties(svyj) //program for calculating the difference between summary measures of the distribution and jackknifing the 95% CI
syntax varlist [if] [pweight iweight/] //specify standard syntax
tokenize `varlist' //sets `1', `2'

ineqdec0 `1' `if' [aweight = `exp']
matrix gini_1 = r(gini)
matrix p90p10_1 = r(p90p10)
matrix p75p25_1 = r(p75p25)
matrix p25p50_1 = r(p25p50)
matrix p10p50_1 = r(p10p50)
matrix p90p50_1 = r(p90p50)
matrix p75p50_1 = r(p75p50)
matrix ge2_1 = r(ge2) 

ineqdec0 `2'  `if' [aweight = `exp']
matrix gini_2 = r(gini)
matrix p90p10_2 = r(p90p10)
matrix p75p25_2 = r(p75p25)
matrix p25p50_2 = r(p25p50)
matrix p10p50_2 = r(p10p50)
matrix p90p50_2 = r(p90p50)
matrix p75p50_2 = r(p75p50)
matrix ge2_2 = r(ge2)
scalar C = r(N)

mean `1' `if' [pweight = `exp']
matrix mean_1 = e(b)
matrix st_dev_1 = e(sd)

mean `2' `if' [pweight = `exp']
matrix mean_2 = e(b)
matrix st_dev_2 = e(sd)

matrix gini_diff = gini_2 - gini_1
matrix ge2_diff = ge2_2 - ge2_1
matrix abs_gini_diff = gini_2 * mean_2 - gini_1 * mean_1
matrix st_dev_diff = st_dev_2 - st_dev_1
matrix p90p10_diff = p90p10_2 - p90p10_1
matrix p75p25_diff = p75p25_2 - p75p25_1
matrix p25p50_diff = p25p50_2 - p25p50_1
matrix p10p50_diff = p10p50_2 - p10p50_1
matrix p90p50_diff = p90p50_2 - p90p50_1
matrix p75p50_diff = p75p50_2 - p75p50_1

gen sample = 0
replace sample = 1 `if' & `exp' != 0 & `2' != .

matrix b = (gini_diff, ge2_diff, abs_gini_diff, st_dev_diff, p90p10_diff, p75p25_diff, p25p50_diff, p10p50_diff, p90p50_diff, p75p50_diff)

matrix colnames b = gini_diff ge2_diff abs_gini_diff st_dev_diff p90p10_diff p75p25_diff p25p50_diff p10p50_diff p90p50_diff p75p50_diff

matrix list b

ereturn post b, esample(sample)
ereturn scalar N = C

end

*****************************************
*** Compute jackknifed wealth measures **
*****************************************

local wealth_waves 2 6 10 14
foreach x of local wealth_waves{
	capture noisily svyset [pweight = weight_`x'], jkrweight(r_weight_`x'_*, multiplier(0.977777)) vce(jackknife) mse
	capture noisily svy jackknife _b: jackknife_diff net_worth_e net_worth_bequest_e if wave == `x'
}







